mod deltastepping;
use deltastepping::*;

use std::{collections::HashSet, sync::{Arc, Mutex}};
use petgraph::graph::UnGraph;

mod inputs;


fn main() {
    delta_stepping(8, false, &inputs::mk_large(), 0.7, 0);
}

#[cfg(test)]
mod test {
    use petgraph::{algo::dijkstra, graph::NodeIndex};

    use super::*;

    fn create_sample1() -> petgraph::Graph<String, f32, petgraph::Undirected> {
        let mut sample1 = UnGraph::<String, f32>::new_undirected();

        for label in "ABCDEFG".chars() {
            sample1.add_node(label.to_string());
        }

        sample1.extend_with_edges(&[
            (0, 1, 3.0), (0, 3, 5.0), (0, 4, 3.0), (0, 6, 3.0), // A
            (1, 0, 3.0), (1, 2, 3.0),                           // B
            (2, 1, 3.0), (2, 3, 1.0),                           // C
            (3, 2, 1.0), (3, 0, 5.0),                           // D
            (4, 0, 3.0), (4, 5, 5.0),                           // E
            (5, 4, 5.0),                                        // F
            (6, 0, 3.0)                                         // G
        ]);

        return sample1;
    }

    
    #[test]
    fn test_1_sample1() {
        let sample1 = inputs::mk_sample1();
        let dst_nodes: Vec<u32> = (0..sample1.node_count() as u32).collect();

        test_sssp(&sample1, 3.0, 0, &dst_nodes);
    }
    #[test]
    fn test_2_sample2() {
        let sample2 = inputs::mk_sample2();
        let dst_nodes: Vec<u32> = (0..sample2.node_count() as u32).collect();

        test_sssp(&sample2, 3.0, 0, &dst_nodes);
    }
    #[test]
    fn test_3_large() {
        let large = inputs::mk_large();
        let dst_nodes: Vec<u32> = (0..100).collect();

        test_sssp(&large, 0.7, 0, &dst_nodes);
    }

    fn test_sssp(
        graph: &petgraph::Graph<String, f32, petgraph::Undirected>,
        delta: f32,
        src: u32,
        dst_nodes: &[u32]
    ) {
        for thread_count in [1, 2, 4, 8] {
            let ret_dists = delta_stepping(thread_count, false, &graph, delta, src);

            let actual: Vec<Option<f32>> = dst_nodes.iter()
                .map(|&d| {
                    let val = ret_dists[d as usize];
                    if val.is_infinite() {None}
                    else { Some(val) }
                }).collect();
            
            let expected: Vec<Option<f32>> = dst_nodes.iter()
                .map(|&d| {
                    dijkstra(
                        graph, 
                        NodeIndex::new(src as usize), 
                        None::<NodeIndex>, 
                        |edge| *edge.weight()
                    ).get(&NodeIndex::new(d as usize)).copied()
                }).collect();

            assert_eq!(actual.len(), expected.len());

            for ((&d, a), e) in dst_nodes.iter().zip(actual.iter()).zip(expected.iter()) {
                match (a, e) {
                    (None, None) => {}
                    (Some(da), Some(de)) => {
                        assert!(
                            (da - de).abs() <= 1e-5,
                            "wrong {d}: got {da}, expected {de}",
                        );
                    }
                    _ => panic!(
                        "got nothing {d}: got {:?}, expected {:?}",
                        a,
                        e
                    ),
                }
            }
        }
    }

    #[test]
    fn z_bench() {
        let graph = inputs::mk_large();
        let run = |threads: u32| {
            delta_stepping(threads as usize, false, &graph, 0.7, 0);
        };
        println!("");
        let n1 = bench("N1", 5, || { run(1); });
        let n2 = bench("N2", 5, || { run(2); });
        let n4 = bench("N4", 5, || { run(4); });
        let n8 = bench("N8", 5, || { run(8); });
        assert!(n1 * 0.4 <= n2 && n2 <= n1 * 0.8); 
        assert!(n1 * 0.2 <= n4 && n4 <= n1 * 0.6); 
        assert!(n1 * 0.1 <= n8 && n8 <= n1 * 0.5); 
    }

    fn bench<F: Fn() -> ()>(label: &str, repeats: usize, f: F) -> f64 {
        let start = std::time::Instant::now();
        for _ in 0 .. repeats {
            f();
        }
        let time = start.elapsed() / repeats as u32;
        println!("     {} {:?}", label, time);
        time.as_nanos() as f64
    }
}